#include "XMLReader.h"
#include "fontchars.h"
#include "tools.h"
#include <xtl.h>

XMLReader* LoadConfigFile(wstring configfile, wstring path)
{
	wstring fullname = configfile;

	if (path.size() > 0)
		fullname = path + L"\\" + configfile;

	/*if (!FileExists(fullname))
	{
		fullname = GetAppPath() + "\\config\\"+configfile;
	}

	if (!FileExists(fullname))
	{
		fullname = "sysconf\\"+configfile;
	}*/

	if (!FileExists(fullname)) return NULL;

	byte * buffer;

	FILE* f;
	_wfopen_s(&f, fullname.c_str(),L"rb");
	fseek (f , 0 , SEEK_END);
	int size = ftell(f);
	fseek(f, 0, 0);
	buffer = (byte*)malloc(size+2);
	//buffer = new WCHAR[size+2];
	int offset = 0;
	int count = 0;
	int toread = 0;
	while( size > 0 )
	{
		toread = min(size,4096);
		// Attempt to read in 100 bytes:
		count = (int)fread( buffer+offset, 1, toread, f );
		if( ferror( f ) )      {
			SAFE_DELETE(buffer);
			return NULL;
		}
		offset += count;
		size -= count;
	}
	fclose(f);
	buffer[offset] = 0;
	buffer[offset+1] = 0;

	XMLReader* xml = new XMLReader();
	wcscpy_s(xml->filename,100,configfile.c_str());

	if (buffer[0] == 0xFF && buffer[1] == 0xFE)
	{
		WCHAR * wbuf = (WCHAR*)buffer+1;
		xml->LoadXML(wbuf);
	} else {
		// non unicoe, read type from xml head
		char * buffer2 = (char*)buffer;

		// get encoding type
		char temp[250];
		memcpy(&temp,buffer2,250);
		temp[249] = 0;
		string temp2 = make_lowercaseA(temp);

		// utf-8 type, convert to unicode
		if (temp2.find("utf-8"))
		{
			int neededbuffer = MultiByteToWideChar(CP_UTF8,0,buffer2,-1,NULL,0);
			if (neededbuffer == 0)
				return NULL;
			WCHAR* buf2 = new WCHAR[neededbuffer];
			memset(buf2,0,neededbuffer);
			int res = MultiByteToWideChar(CP_UTF8,0,buffer2,-1,buf2,neededbuffer);
			if (res == 0)
				return NULL;

			delete buffer;
			xml->LoadXML(buf2);
		} else {

		}
	}

	return xml;
}

XMLReader* XMLFromString(wstring & input)
{
	XMLReader * xml = new XMLReader();
	WCHAR * chardata = new WCHAR[input.size()+1];
	wcscpy_s(chardata,input.size()+1,input.c_str());
	xml->LoadXML(chardata);

	return xml;
}

XMLReader::XMLReader()
{
	createSpecialCharacterList();
}

XMLReader::~XMLReader(void)
{
}

//! Constructor
void XMLReader::LoadXML(const WCHAR* srcbuf, ETEXT_FORMAT in_SourceFormat, ETEXT_FORMAT in_TargetFormat)
/*	: TextData(0), P(0), TextSize(0), TextBegin(0), CurrentNodeType(EXN_NONE),
	SourceFormat(ETF_ASCII), TargetFormat(ETF_ASCII)*/
{
	TextData = 0;
	P = 0;
	TextSize = 0;
	TextBegin = 0;
	CurrentNodeType = EXN_NONE;
	SourceFormat = in_SourceFormat;
	TargetFormat = in_TargetFormat;

	int size = (int)wcslen(srcbuf);
	TextBegin = (WCHAR*)srcbuf;
	TextData = (WCHAR*)srcbuf;
	TextSize = size;


	// set pointer to text begin
	P = TextBegin;
}


bool XMLReader::read()
{
	// if not end reached, parse the node
	if (P && (unsigned int)(P - TextBegin) < TextSize - 1 && *P != 0)
	{
		parseCurrentNode();
		return true;
	}

	return false;
}

void XMLReader::restart()
{
	P = TextBegin;
}

int XMLReader::GetOffset()
{
	return P - TextBegin;
}

int XMLReader::GetSize()
{
	return TextSize;
}

//! Returns the type of the current XML node.
EXML_NODE XMLReader::getNodeType()
{
	return CurrentNodeType;
}

//! Returns attribute count of the current XML node.
int XMLReader::getAttributeCount()
{
	return (int)Attributes.size();
}


//! Returns name of an attribute.
const WCHAR* XMLReader::getAttributeName(int idx)
{
	if (idx < 0 || idx >= (int)Attributes.size())
		return 0;

	return Attributes[idx].Name.c_str();
}


//! Returns the value of an attribute. 
const WCHAR* XMLReader::getAttributeValue(int idx)
{
	if (idx < 0 || idx >= (int)Attributes.size())
		return 0;

	return Attributes[idx].Value.c_str();
}


//! Returns the value of an attribute. 
const WCHAR* XMLReader::getAttributeValue(const WCHAR* name)
{
	SAttribute* attr = getAttributeByName(name);
	if (!attr)
		return EmptyString.c_str();

	return attr->Value.c_str();
}

//! Returns the value of an attribute. 
const WCHAR* XMLReader::getAttributeValue(const WCHAR* name, const WCHAR * defval)
{
	SAttribute* attr = getAttributeByName(name);
	if (!attr)
		return defval;

	return attr->Value.c_str();
}

//! Returns the value of an attribute
const WCHAR* XMLReader::getAttributeValueSafe(const WCHAR* name)
{
	SAttribute* attr = getAttributeByName(name);
	if (!attr)
		return EmptyString.c_str();

	return attr->Value.c_str();
}



//! Returns the value of an attribute as integer. 
int XMLReader::getAttributeValueAsInt(const WCHAR* name)
{
	const SAttribute* attr = getAttributeByName(name);
	if (!attr)
		return 0;

	return _wtoi(attr->Value.c_str());
}

bool XMLReader::getAttributeValueAsBool(const WCHAR* name)
{
	return (int)getAttributeValueAsInt(name)!=0;
}


//! Returns the value of an attribute as integer. 
int XMLReader::getAttributeValueAsInt(int idx)
{
	const WCHAR* attrvalue = getAttributeValue(idx);
	if (!attrvalue)
		return 0;

	return _wtoi(attrvalue);
}


//! Returns the value of an attribute as float. 
float XMLReader::getAttributeValueAsFloat(const WCHAR* name)
{
	const SAttribute* attr = getAttributeByName(name);
	if (!attr)
		return 0;

	wstring c = attr->Value.c_str();
	return fast_atof(c.c_str());
}


//! Returns the value of an attribute as float. 
float XMLReader::getAttributeValueAsFloat(int idx)
{
	const WCHAR* attrvalue = getAttributeValue(idx);
	if (!attrvalue)
		return 0;

	wstring c = attrvalue;
	return fast_atof(c.c_str());
}


//! Returns the name of the current node.
const WCHAR* XMLReader::getNodeName()
{
	return NodeName.c_str();
}


//! Returns data of the current node.
const WCHAR* XMLReader::getNodeData()
{
	return NodeName.c_str();
}

//! Returns data of the current node.
wstring XMLReader::getSubNodeData()
{
	read();
	if (CurrentNodeType == EXN_TEXT)
		return NodeName;

	return L"";
}


//! Returns if an element is an empty element, like <foo />
bool XMLReader::isEmptyElement()
{
	return IsEmptyElement;
}

//! Returns format of the source xml file.
ETEXT_FORMAT XMLReader::getSourceFormat()
{
	return SourceFormat;
}

//! Returns format of the strings returned by the parser.
ETEXT_FORMAT XMLReader::getParserFormat()
{
	return TargetFormat;
}


// Reads the current xml node
void XMLReader::parseCurrentNode()
{
	WCHAR* start = P;

	// more forward until '<' found
	while(*P != L'<' && *P)
		++P;

	if (!*P)
		return;

	if (P - start > 0)
	{
		// we found some text, store it
		if (setText(start, P))
			return;
	}

	++P;

	// based on current token, parse and report next element
	switch(*P)
	{
	case L'/':
		parseClosingXMLElement(); 
		break;
	case L'?':
		ignoreDefinition();	
		break;
	case L'!':
		if (!parseCDATA())
			parseComment();	
		break;
	default:
		parseOpeningXMLElement();
		break;
	}
}


//! sets the state that text was found. Returns true if set should be set
bool XMLReader::setText(WCHAR* start, WCHAR* end)
{
	// check if text is more than 2 characters, and if not, check if there is 
	// only white space, so that this text won't be reported
	if (end - start < 3)
	{
		WCHAR* p = start;
		for(; p != end; ++p)
			if (!isWhiteSpace(*p))
				break;

		if (p == end)
			return false;
	}

	// set current text to the parsed text, and replace xml special characters
	wstring s(start, (int)(end - start));
	NodeName = replaceSpecialCharacters(s);

	// current XML node type is text
	CurrentNodeType = EXN_TEXT;

	return true;
}



//! ignores an xml definition like <?xml something />
void XMLReader::ignoreDefinition()
{
	CurrentNodeType = EXN_UNKNOWN;

	// move until end marked with '>' reached
	while(*P != L'>')
		++P;

	++P;
}


//! parses a comment
void XMLReader::parseComment()
{
	CurrentNodeType = EXN_COMMENT;
	P += 1;

	WCHAR *pCommentBegin = P;

	int count = 1;

	// move until end of comment reached
	while(count)
	{
		if (*P == L'>')
			--count;
		else
		if (*P == L'<')
			++count;

		++P;
	}

	P -= 3;
	NodeName = wstring(pCommentBegin+2, (int)(P - pCommentBegin-2));
	P += 3;
}


//! parses an opening xml element and reads attributes
void XMLReader::parseOpeningXMLElement()
{
	CurrentNodeType = EXN_ELEMENT;
	IsEmptyElement = false;
	Attributes.clear();

	// find name
	const WCHAR* startName = P;

	// find end of element
	while(*P != L'>' && !isWhiteSpace(*P))
		++P;

	const WCHAR* endName = P;

	// find Attributes
	while(*P != L'>')
	{
		if (isWhiteSpace(*P))
			++P;
		else
		{
			if (*P != L'/')
			{
				// we've got an attribute

				// read the attribute names
				const WCHAR* attributeNameBegin = P;

				while(!isWhiteSpace(*P) && *P != L'=')
					++P;

				const WCHAR* attributeNameEnd = P;
				++P;

				// read the attribute value
				// check for quotes and single quotes, thx to murphy
				while( (*P != L'\"') && (*P != L'\'') && *P) 
					++P;

				if (!*P) // malformatted xml file
					return;

				const WCHAR attributeQuoteChar = *P;

				++P;
				const WCHAR* attributeValueBegin = P;
				
				while(*P != attributeQuoteChar && *P)
					++P;

				if (!*P) // malformatted xml file
					return;

				const WCHAR* attributeValueEnd = P;
				++P;

				SAttribute attr;
				attr.Name = wstring(attributeNameBegin, 
					(int)(attributeNameEnd - attributeNameBegin));

				wstring s(attributeValueBegin, 
					(int)(attributeValueEnd - attributeValueBegin));

				attr.Value = replaceSpecialCharacters(s);
				Attributes.push_back(attr);
			}
			else
			{
				// tag is closed directly
				++P;
				IsEmptyElement = true;
				break;
			}
		}
	}

	// check if this tag is closing directly
	if (endName > startName && *(endName-1) == L'/')
	{
		// directly closing tag
		IsEmptyElement = true;
		endName--;
	}
	
	NodeName = wstring(startName, (int)(endName - startName));

	++P;
}


//! parses an closing xml tag
void XMLReader::parseClosingXMLElement()
{
	CurrentNodeType = EXN_ELEMENT_END;
	IsEmptyElement = false;
	Attributes.clear();

	++P;
	const WCHAR* pBeginClose = P;

	while(*P != L'>')
		++P;

	NodeName = wstring(pBeginClose, (int)(P - pBeginClose));
	++P;
}

//! parses a possible CDATA section, returns false if begin was not a CDATA section
bool XMLReader::parseCDATA()
{
	if (*(P+1) != L'[')
		return false;

	CurrentNodeType = EXN_CDATA;

	// skip '<![CDATA['
	int count=0;
	while( *P && count<8 )
	{
		++P;
		++count;
	}

	if (!*P)
		return true;

	WCHAR *cDataBegin = P;
	WCHAR *cDataEnd = 0;

	// find end of CDATA
	while(*P && !cDataEnd)
	{
		if (*P == L'>' && 
		   (*(P-1) == L']') &&
		   (*(P-2) == L']'))
		{
			cDataEnd = P - 2;
		}

		++P;
	}

	if ( cDataEnd )
		NodeName = wstring(cDataBegin, (int)(cDataEnd - cDataBegin));
	else
		NodeName = L"";

	return true;
}

// finds a current attribute by name, returns 0 if not found
SAttribute* XMLReader::getAttributeByName(const WCHAR* name)
{
	if (!name)
		return 0;

	wstring n = name;

	for (int i=0; i<(int)Attributes.size(); ++i)
		if (Attributes[i].Name == n)
			return &Attributes[i];

	return 0;
}

// replaces xml special characters in a string and creates a new one
wstring XMLReader::replaceSpecialCharacters(
	wstring& origstr)
{
	//int pos = origstr.findFirst(L'&');
	int pos = (int)origstr.find('&');
	int oldPos = 0;

	if (pos == -1)
		return origstr;

	wstring newstr;

	while(pos != -1 && pos < (int)origstr.size()-2)
	{
		// check if it is one of the special characters

		int specialChar = -1;
		for (int i=0; i<(int)SpecialCharacters.size(); ++i)
		{
			const WCHAR* p = &origstr.c_str()[pos]+1;

			if (equalsn(&SpecialCharacters[i][1], p, (int)SpecialCharacters[i].size()-1))
			{
				specialChar = i;
				break;
			}
		}

		if (specialChar != -1)
		{
			newstr += origstr.substr(oldPos, pos - oldPos);
			newstr += SpecialCharacters[specialChar][0];
			pos += (int)SpecialCharacters[specialChar].size();
		}
		else
		{
			newstr.append(origstr.substr(oldPos, pos - oldPos + 1));
			pos += 1;
		}

		// find next &
		oldPos = pos;
		pos = (int)origstr.find('&', pos);		
	}

	if (oldPos < (int)origstr.size())
		newstr.append(origstr.substr(oldPos, origstr.size()-oldPos));

	return newstr;
}

//! returns if a format is little endian
bool XMLReader::isLittleEndian(ETEXT_FORMAT f)
{
	return f == ETF_ASCII ||
	       f == ETF_UTF8 ||
	       f == ETF_UTF16_LE ||
	       f == ETF_UTF32_LE;
}

//! returns true if a character is whitespace
bool XMLReader::isWhiteSpace(WCHAR c)
{
	return (c==L' ' || c==L'\t' || c==L'\n' || c==L'\r');
}


//! generates a list with xml special characters
void XMLReader::createSpecialCharacterList()
{
	// list of strings containing special symbols, 
	// the first character is the special character,
	// the following is the symbol string without trailing &.

	SpecialCharacters.push_back(L"&amp;");
	SpecialCharacters.push_back(L"<lt;");
	SpecialCharacters.push_back(L">gt;");
	SpecialCharacters.push_back(L"\"quot;");
	SpecialCharacters.push_back(L"'apos;");

	GetXMLSpecialChars(SpecialCharacters);
}

//! compares the first n characters of the strings
bool XMLReader::equalsn(const WCHAR* str1, const WCHAR* str2, int len)
{
	int i;
	for(i=0; str1[i] && str2[i] && i < len; ++i)
		if (str1[i] != str2[i])
			return false;

	// if one (or both) of the strings was smaller then they
	// are only equal if they have the same lenght
	return (i == len) || (str1[i] == 0 && str2[i] == 0);
}


bool StartAttribute(XMLReader* xml, wstring attribute)
{
	if (xml->getNodeType() != EXN_ELEMENT) return false;
	wstring name = xml->getNodeName();
	if (name == attribute) return true;
	return false;
}

bool EndAttribute(XMLReader* xml, wstring attribute)
{
	wstring name = xml->getNodeName();
	if (xml->isEmptyElement() && attribute == name)
	{
		return true;
	}
	if (!xml->read())
		return true;

	if (xml->getNodeType() != EXN_ELEMENT_END) return false;
	name = xml->getNodeName();
	if (name == attribute) return true;
	return false;
}

void FreeXMLAndText(XMLReader * xml)
{
	if (xml)
	{
		SAFE_DELETE_A(xml->TextBegin);
		SAFE_DELETE(xml);
	}
}

void FreeXML(XMLReader * xml)
{
	SAFE_DELETE(xml);
}